-- 医院电商系统数据库表结构

-- 1. 产品分类表
CREATE TABLE IF NOT EXISTS product_category (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    category_code VARCHAR(50) UNIQUE NOT NULL COMMENT '分类编码',
    category_name VARCHAR(100) NOT NULL COMMENT '分类名称',
    parent_id BIGINT DEFAULT 0 COMMENT '父分类ID，0表示顶级分类',
    category_type VARCHAR(20) NOT NULL COMMENT '分类类型：SERVICE-服务性产品，MEDICAL-医疗器械',
    sort_order INT DEFAULT 0 COMMENT '排序序号',
    description TEXT COMMENT '分类描述',
    icon_url VARCHAR(500) COMMENT '分类图标URL',
    status TINYINT DEFAULT 1 COMMENT '状态：1-启用，0-禁用',
    hospital_id VARCHAR(50) COMMENT '医院ID（可选，支持多医院）',
    create_time DATETIME DEFAULT CURRENT_TIMESTAMP,
    update_time DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_parent_id (parent_id),
    INDEX idx_category_type (category_type),
    INDEX idx_hospital_id (hospital_id)
) COMMENT '产品分类表';

-- 2. 产品表
CREATE TABLE IF NOT EXISTS product (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    product_code VARCHAR(50) UNIQUE NOT NULL COMMENT '产品编码',
    product_name VARCHAR(200) NOT NULL COMMENT '产品名称',
    category_id BIGINT NOT NULL COMMENT '分类ID',
    product_type VARCHAR(20) NOT NULL COMMENT '产品类型：SERVICE-服务性产品，MEDICAL-医疗器械',
    price DECIMAL(10,2) NOT NULL COMMENT '价格',
    original_price DECIMAL(10,2) COMMENT '原价（用于显示折扣）',
    unit VARCHAR(20) DEFAULT '件' COMMENT '单位：件、次、套等',
    stock_quantity INT DEFAULT 0 COMMENT '库存数量（服务性产品可为-1表示无限制）',
    description TEXT COMMENT '产品描述',
    detail_html TEXT COMMENT '产品详情HTML',
    main_image_url VARCHAR(500) COMMENT '主图URL',
    image_urls TEXT COMMENT '产品图片URLs（JSON数组）',
    specifications TEXT COMMENT '规格参数（JSON格式）',
    status TINYINT DEFAULT 1 COMMENT '状态：1-上架，0-下架',
    is_recommended TINYINT DEFAULT 0 COMMENT '是否推荐：1-是，0-否',
    sort_order INT DEFAULT 0 COMMENT '排序序号',
    hospital_id VARCHAR(50) COMMENT '医院ID',
    sales_count INT DEFAULT 0 COMMENT '销售数量',
    view_count INT DEFAULT 0 COMMENT '浏览次数',
    create_time DATETIME DEFAULT CURRENT_TIMESTAMP,
    update_time DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_category_id (category_id),
    INDEX idx_product_type (product_type),
    INDEX idx_status (status),
    INDEX idx_hospital_id (hospital_id),
    FOREIGN KEY (category_id) REFERENCES product_category(id)
) COMMENT '产品表';

-- 3. 购物车表
CREATE TABLE IF NOT EXISTS shopping_cart (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id VARCHAR(50) NOT NULL COMMENT '用户ID（患者ID或医院用户ID）',
    user_type VARCHAR(20) NOT NULL COMMENT '用户类型：PATIENT-患者，HOSPITAL-医院用户',
    product_id BIGINT NOT NULL COMMENT '产品ID',
    quantity INT NOT NULL DEFAULT 1 COMMENT '数量',
    hospital_id VARCHAR(50) COMMENT '医院ID',
    create_time DATETIME DEFAULT CURRENT_TIMESTAMP,
    update_time DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY uk_user_product (user_id, product_id),
    INDEX idx_user_id (user_id),
    INDEX idx_product_id (product_id),
    INDEX idx_hospital_id (hospital_id),
    FOREIGN KEY (product_id) REFERENCES product(id)
) COMMENT '购物车表';

-- 4. 订单表
CREATE TABLE IF NOT EXISTS product_order (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    order_no VARCHAR(50) UNIQUE NOT NULL COMMENT '订单号',
    user_id VARCHAR(50) NOT NULL COMMENT '用户ID',
    user_type VARCHAR(20) NOT NULL COMMENT '用户类型：PATIENT-患者，HOSPITAL-医院用户',
    hospital_id VARCHAR(50) COMMENT '医院ID',
    total_amount DECIMAL(10,2) NOT NULL COMMENT '订单总金额',
    discount_amount DECIMAL(10,2) DEFAULT 0 COMMENT '优惠金额',
    pay_amount DECIMAL(10,2) NOT NULL COMMENT '实付金额',
    order_status VARCHAR(20) NOT NULL COMMENT '订单状态：PENDING-待支付，PAID-已支付，SHIPPED-已发货，DELIVERED-已送达，COMPLETED-已完成，CANCELLED-已取消',
    payment_status VARCHAR(20) DEFAULT 'UNPAID' COMMENT '支付状态：UNPAID-未支付，PAID-已支付，REFUNDED-已退款',
    payment_method VARCHAR(20) COMMENT '支付方式：ALIPAY-支付宝，WECHAT-微信，BALANCE-余额等',
    payment_time DATETIME COMMENT '支付时间',
    shipping_address TEXT COMMENT '收货地址（JSON格式）',
    contact_name VARCHAR(50) COMMENT '联系人姓名',
    contact_phone VARCHAR(20) COMMENT '联系电话',
    remark TEXT COMMENT '订单备注',
    create_time DATETIME DEFAULT CURRENT_TIMESTAMP,
    update_time DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_user_id (user_id),
    INDEX idx_order_no (order_no),
    INDEX idx_order_status (order_status),
    INDEX idx_hospital_id (hospital_id),
    INDEX idx_create_time (create_time)
) COMMENT '订单表';

-- 5. 订单明细表
CREATE TABLE IF NOT EXISTS product_order_item (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    order_id BIGINT NOT NULL COMMENT '订单ID',
    order_no VARCHAR(50) NOT NULL COMMENT '订单号',
    product_id BIGINT NOT NULL COMMENT '产品ID',
    product_code VARCHAR(50) COMMENT '产品编码（快照）',
    product_name VARCHAR(200) COMMENT '产品名称（快照）',
    product_image_url VARCHAR(500) COMMENT '产品图片（快照）',
    price DECIMAL(10,2) NOT NULL COMMENT '单价（快照）',
    quantity INT NOT NULL COMMENT '数量',
    subtotal DECIMAL(10,2) NOT NULL COMMENT '小计金额',
    create_time DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_order_id (order_id),
    INDEX idx_order_no (order_no),
    INDEX idx_product_id (product_id),
    FOREIGN KEY (order_id) REFERENCES product_order(id),
    FOREIGN KEY (product_id) REFERENCES product(id)
) COMMENT '订单明细表';

-- 6. 服务商户配置表（支付配置）
CREATE TABLE IF NOT EXISTS service_merchant_config (
    x_id BIGINT PRIMARY KEY AUTO_INCREMENT COMMENT 'id',
    x_create_time DATETIME DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
    x_update_time DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '更新时间',
    x_remark VARCHAR(255) DEFAULT NULL COMMENT '备注',
    status INT(8) DEFAULT NULL COMMENT '状态',
    merchant_seq VARCHAR(255) DEFAULT NULL COMMENT '对应应用code',
    app_code VARCHAR(255) DEFAULT NULL COMMENT '节点code',
    organ_name VARCHAR(255) DEFAULT NULL COMMENT '医院首字母',
    biz_sys_seq VARCHAR(255) DEFAULT NULL COMMENT '业务系统编码',
    apply_id VARCHAR(255) DEFAULT NULL COMMENT '应用id',
    apply_key VARCHAR(255) DEFAULT NULL COMMENT '应用key',
    apply_name VARCHAR(255) DEFAULT NULL COMMENT '应用名称',
    organ_id INT(11) DEFAULT NULL COMMENT '医院id',
    INDEX idx_organ_id (organ_id),
    INDEX idx_app_code (app_code),
    INDEX idx_status (status)
) COMMENT '服务商户配置表（支付配置）';

-- 7. 服务支付账单表
CREATE TABLE IF NOT EXISTS service_pay_bill (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    status INT DEFAULT NULL COMMENT '状态：3-支付成功，6-退款成功',
    app_code VARCHAR(255) DEFAULT NULL COMMENT '应用code',
    organ_name VARCHAR(255) DEFAULT NULL COMMENT '医院名称',
    work_service_code VARCHAR(255) DEFAULT NULL COMMENT '工作服务编码',
    apply_code VARCHAR(255) DEFAULT NULL COMMENT '应用编码',
    pay_channel VARCHAR(50) DEFAULT NULL COMMENT '支付渠道',
    trade_time DATETIME DEFAULT NULL COMMENT '交易时间',
    trade_no VARCHAR(255) DEFAULT NULL COMMENT '渠道流水号',
    out_trade_no VARCHAR(255) DEFAULT NULL COMMENT '商户订单号（订单号）',
    trade_status VARCHAR(50) DEFAULT NULL COMMENT '交易状态',
    order_amount DECIMAL(10,2) DEFAULT NULL COMMENT '订单金额',
    deal_amount DECIMAL(10,2) DEFAULT NULL COMMENT '实际交易金额',
    service_code VARCHAR(255) DEFAULT NULL COMMENT '服务编码',
    deal_trade_no VARCHAR(255) DEFAULT NULL COMMENT '平台订单号',
    refund_no VARCHAR(255) DEFAULT NULL COMMENT '退款单号',
    out_refund_no VARCHAR(255) DEFAULT NULL COMMENT '商户退款单号',
    refund_time DATETIME DEFAULT NULL COMMENT '退款时间',
    refund_amount DECIMAL(10,2) DEFAULT NULL COMMENT '退款金额',
    remark VARCHAR(500) DEFAULT NULL COMMENT '备注',
    create_time DATETIME DEFAULT CURRENT_TIMESTAMP,
    update_time DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_out_trade_no (out_trade_no),
    INDEX idx_status (status),
    INDEX idx_trade_no (trade_no),
    INDEX idx_create_time (create_time)
) COMMENT '服务支付账单表';
